<?php
ini_set('memory_limit', '6G');


class ParsingTool
{
    const FEED_FILE_MASK = '/autoshopper_[^_dealerinfo]+.csv/';
    const FILE_DEALER_INFO = '/AutoXLooCRM.csv';

    /**
     * Flag when Dealer info was added
     * @var bool
     */
    protected $updated = false;

    /**
     * @var
     */
    protected $feedArray;

    /**
     * @var
     */
    protected $sourceFeedFile;

    /**
     * @var
     */
    protected $dealerInfoArray;

    /**
     * ParsingTool constructor.
     */
    public function __construct()
    {
        $this->loadDealerInfo();
        $this->processFiles();
    }

    /**
     * Load Dealer info
     */
    protected function loadDealerInfo()
    {
        if (file_exists(__DIR__ . self::FILE_DEALER_INFO)
            && ($handle = fopen(__DIR__ . self::FILE_DEALER_INFO,'r')) !== false)
        {
            $captions = [];
            $captionCustomerKey = false;
            while (($data = fgetcsv($handle, null, ',')) !== false) {
                if (empty($captions)) {
                    $captions = $data;
                    $captionCustomerKey = array_search('﻿Customer', $captions);
                } else {
                    $this->dealerInfoArray[$data[$captionCustomerKey]] = array_combine($captions, $data);
                    $this->dealerInfoArray[$data[$captionCustomerKey]]['DealerAddress'] = '';
                    $this->dealerInfoArray[$data[$captionCustomerKey]]['DealerCity'] = '';
                    $this->dealerInfoArray[$data[$captionCustomerKey]]['DealerState'] = '';
                    $this->dealerInfoArray[$data[$captionCustomerKey]]['DealerZIP'] = '';
                    $this->dealerInfoArray[$data[$captionCustomerKey]]['DealerPhone'] = '';
                    $this->dealerInfoArray[$data[$captionCustomerKey]]['DealerEmail'] = '';
                }
            }

            fclose($handle);
        }
    }

    /**
     * Parse feed and save to file
     */
    protected function processFiles()
    {
        $files = glob(__DIR__ . '/*');
        foreach ($files as $fileName) {
            if (preg_match(self::FEED_FILE_MASK, $fileName)) {
                $this->processFeed($fileName);
                $this->saveCsvFile();
            }
        }
    }

    /**
     * Parse CSV file and add Dealer's information
     * @param $fileName
     * @return void
     */
    protected function processFeed($fileName)
    {
        $this->sourceFeedFile = $fileName;
        if (file_exists($fileName)
            && ($handle = fopen($fileName,'r')) !== false)
        {
            $captions = [];
            while (($data = fgetcsv($handle, null, ',')) !== false) {
                if (empty($captions)) {
                    $captions = $data;
                } else {
                    $diff = array_diff_key($captions, $data);
                    if ($diff) {
                        $data[] = $diff;
                    }
                    $line = array_combine($captions, $data);
                    $companyName = $line['CompanyName'];
                    if ($companyName && $this->dealerInfoArray[$companyName]) {
                        $this->updated = true;
                    }
                    $line['DealerAddress'] = @$this->dealerInfoArray[$companyName]['Address 1'];
                    $line['DealerCity'] = @$this->dealerInfoArray[$companyName]['City'];
                    $line['DealerState'] = @$this->dealerInfoArray[$companyName]['State/Region'];
                    $line['DealerZIP'] = @$this->dealerInfoArray[$companyName]['Zip/Postal Code'];
                    $line['DealerPhone'] = @$this->dealerInfoArray[$companyName]['Phone'];
                    $line['DealerEmail'] = @$this->dealerInfoArray[$companyName]['Email'];
                    $this->feedArray[] = $line;
                }
            }
            fclose($handle);
        }
    }

    /**
     * Save data to CSV file
     */
    protected function saveCsvFile()
    {
        if ($this->updated) {
            $targetFileName = substr($this->sourceFeedFile, 0, -4) . '_dealerinfo.csv';
            $handle = fopen($targetFileName, 'w');
            $captions = [];
            foreach ($this->feedArray as $feed) {
                if (empty($captions)) {
                    $captions = array_keys($feed);
                    fputs($handle, '"' . join('","', $captions) . '"' . "\r\n");
                }
                fputs($handle, '"' . join('","', $feed) . '"' . "\r\n");
            }
            fclose($handle);
            $this->updated = false;
        }
        $this->feedArray = [];
        $this->sourceFeedFile = '';
    }

}
(new ParsingTool());